require("dotenv").config();
const express = require("express");
const mongoose = require("mongoose");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { auth } = require("./middleware/auth");
const { sendEmail } = require("./middleware/mailer");
const User = require("./models/User");
const Job = require("./models/Job");
const Category = require("./models/Category");
const crypto = require('crypto')
const pdfParse = require("pdf-parse");
const xlsx = require("xlsx");
const mammoth = require("mammoth");
const extractProfileDetails = require("./middleware/cvExtract");
const Application = require("./models/Application");
const { body, validationResult } = require("express-validator");
const cookieParser = require("cookie-parser");
const multer = require("multer")
const path = require('path');
const Grid = require('gridfs-stream');
const GridFsStorage = require('multer-gridfs-storage').GridFsStorage;
const cors = require("cors");
const nodemailer = require("nodemailer");
const { GridFSBucket } = require("mongodb");
const helmet = require("helmet")
const rateLimit=require("express-rate-limit")
const { Readable } = require("stream");
const nurseController= require("./controller/nurseController")
const viewController=require("./controller/viewController")
const filePath = path.join(__dirname, "./uploads/Selected_Applicants (1).xlsx"); 
const workbook = xlsx.readFile(filePath);
const sanitizer= require("express-mongo-sanitize")
const xss=require("xss-clean")
const usersController=require("./controller/usersController")

const app = express();
app.use(express.json());
app.use(cookieParser());
app.use(helmet())
app.use(sanitizer())
app.use(xss())
app.set('trust proxy', process.env.NODE_ENV === 'production' ? 1 : false);

const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 115, // Limit each IP to 100 requests per window
  standardHeaders: true, // Return rate limit info in the `RateLimit-*` headers
  legacyHeaders: false, // Disable the `X-RateLimit-*` headers
  message: "Too many requests from this IP, please try again later.",
});

// Apply the rate limiting middleware to all requests.
app.use("/api", limiter)

const allowedOrigins = [
  "http://localhost:5174",
  "http://localhost:5173",
  "https://amsoljobs.africa", 
  /\.amsoljobs\.africa$/, 
  "https://amsol-api-4.onrender.com", 
];

const corsOptions = {
  origin: (origin, callback) => {
    if (
      allowedOrigins.some((o) =>
        o instanceof RegExp ? o.test(origin) : o === origin
      ) ||
      !origin
    ) {
      callback(null, true);
    } else {
      callback(new Error("Not allowed by CORS"));
    }
  },
  credentials: true, // Allow cookies across domains
  methods: ["GET", "POST", "PUT", "DELETE", "OPTIONS"],
  allowedHeaders: ["Content-Type", "Authorization"],
};

app.use(cors(corsOptions));
app.options("*", (req, res) => {
  res.setHeader("Access-Control-Allow-Origin", req.headers.origin || "*");
  res.setHeader("Access-Control-Allow-Credentials", "true");
  res.setHeader(
    "Access-Control-Allow-Methods",
    "GET, POST, PUT, DELETE, OPTIONS"
  );
  res.setHeader("Access-Control-Allow-Headers", "Content-Type, Authorization");
  res.sendStatus(204); 
});

// Near the top, keep existing imports
// require("dotenv").config();
// const express = require("express");
// const mongoose = require("mongoose");
// ... other imports ...

// ... existing middleware ...

// Database Connection
const mongoURI = process.env.MONGODB_URI;
if (!mongoURI) {
  console.error("MongoDB URI not found in environment variables.");
  process.exit(1);
}

let gfsBucket;

// Connect to MongoDB without deprecated options
mongoose
  .connect(mongoURI, {
    serverSelectionTimeoutMS: 5000, // Timeout for server selection
  })
  .then(() => console.log("Connected to MongoDB"))
  .catch((err) => {
    console.error("MongoDB connection error:", err);
    process.exit(1); // Exit on connection failure
  });

mongoose.connection.once("open", () => {
  gfsBucket = new GridFSBucket(mongoose.connection.db, { bucketName: "uploads" });
  console.log("GridFSBucket initialized for uploads collection.");
});

// Health check endpoint for Railway
app.get("/health", async (req, res) => {
  try {
    await mongoose.connection.db.admin().ping();
    res.status(200).send("OK");
  } catch (err) {
    res.status(500).send("MongoDB connection failed");
  }
});

// ... rest of your routes ...

// Start the server with Railway's PORT
const PORT = process.env.PORT || 5000;
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

// Handle SIGTERM for graceful shutdown
process.on("SIGTERM", async () => {
  console.log("Received SIGTERM. Closing MongoDB connection...");
  await mongoose.connection.close();
  console.log("MongoDB connection closed");
  process.exit(0);
});

const storage = multer.memoryStorage(); 
const upload = multer({ storage });

app.use('/api',  nurseController);
app.use('/api',viewController);
app.use("/api", usersController)
app.post("/upload-excel", upload.single("file"), async (req, res) => {
  if (!req.file) {
    return res.status(400).send("No file uploaded.");
  }

  try {
    // Read the workbook directly from the file buffer
    const workbook = xlsx.read(req.file.buffer, { type: "buffer" });
    const sheetName = workbook.SheetNames[0];
    const sheet = workbook.Sheets[sheetName];
    const data = xlsx.utils.sheet_to_json(sheet);

    // Iterate over each row in the Excel sheet
    for (const row of data) {
      let applicantId = null;
      let jobId = null;

      if (row.applicant) {
        try {
          applicantId = new mongoose.Types.ObjectId(row.applicant);
        } catch (err) {
          console.error("Invalid applicant ID:", row.applicant);
          continue; 
        }
      }

      if (row.job) {
        try {
          jobId = new mongoose.Types.ObjectId(row.job);
        } catch (err) {
          console.error("Invalid job ID:", row.job);
          continue; 
        }
      }

      const workExperience = row["workExperience.company"]
        ? [
            {
              company: row["workExperience.company"] || "",
              position: row["workExperience.position"] || "",
              duration: row["workExperience.duration"] || "",
            },
          ]
        : [];

      const academicLevel = row.academicLevel || "others"; 
      const validAcademicLevels = Application.schema.path("academicLevel").enumValues;
      if (!validAcademicLevels.includes(academicLevel)) {
        console.error("Invalid academic level:", academicLevel);
        continue; 
      }

      // Log row data for debugging
      console.log("Row data:", row);

      // Create new application object with cvFileId set to null
      const newApplication = new Application({
        applicant: applicantId,
        job: jobId,
        resume: row.resume || "",
        coverLetter: row.coverLetter || "",
        firstName: row.firstName || "",
        secondName: row.secondName || "",
        lastName: row.lastName || "",
        idNumber: row.idNumber || "",
        whatsAppNo: row.whatsAppNo || "",
        phoneNumber: row.phoneNumber || "",
        email: row.email || "",
        age: row.age || 0,
        nationality: row.nationality || "",
        location: row.location || "",
        specialization: row.specialization || "",
        academicLevel: academicLevel,
        workExperience: workExperience,
        salaryInfo: row.salaryInfo || "",
        cvFileId: null // Explicitly set to null to avoid casting error
      });

      // Save the application to the database
      await newApplication.save();
    }

    res.send("Applications uploaded successfully.");
  } catch (error) {
    console.error("Error uploading applications:", error);
    res.status(500).send("Error uploading applications.");
  }
});


app.get("/api/check-session", (req, res) => {
  if (req.session && req.session.user) {
    res.json({ loggedIn: true });
  } else {
    res.json({ loggedIn: false });
  }
});

app.put('/api/profile/:userId',auth, upload.single('file'), async (req, res) => {
  try {
    const userId = req.params.userId;

    // Save file to GridFS
    const fileId = new mongoose.Types.ObjectId();
    const uploadStream = gfsBucket.openUploadStreamWithId(fileId, req.file.originalname, {
      contentType: req.file.mimetype,
    });
    uploadStream.end(req.file.buffer);

    // Update the user's profile with the fileId
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ message: 'User not found' });
    }

    user.profile.profilePicture = fileId; // Save the fileId in the user document
    await user.save();

    res.json({ message: 'Profile picture updated', fileId });
  } catch (error) {
    console.error('Error updating profile picture:', error);
    res.status(500).json({ message: 'Error updating profile picture' });
  }
});

app.get("/api/profile/picture/:fileId", async (req, res) => {
  try {
    const fileId = new mongoose.Types.ObjectId(req.params.fileId);

    // Retrieve the file from GridFS
    const file = await gfsBucket.find({ _id: fileId }).toArray();

    if (!file || !file.length) {
      return res.status(404).json({ message: "File not found" });
    }

    // Create a read stream and pipe it to the response
    const readStream =gfsBucket.openDownloadStream(fileId);
    res.set("Content-Type", file[0].contentType);
    readStream.pipe(res);
  } catch (error) {
    console.error("Error fetching profile picture:", error);
    res.status(500).json({ message: "Server error" });
  }
});

app.post(
  "/api/register",
  [
    body("username").notEmpty().withMessage("Username is required"),
    body("email").isEmail().withMessage("Please enter a valid email address"),
    body("password")
      .isLength({ min: 6 }).withMessage("Password must be at least 6 characters long")
      .matches(/\d/).withMessage("Password must contain a number")
      .matches(/[A-Z]/).withMessage("Password must contain an uppercase letter"),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { username, email, password, role } = req.body;
    try {
      const existingUser = await User.findOne({ email });
      if (existingUser) {
        return res.status(400).json({ message: "User already exists" });
      }

      const hashedPassword = await bcrypt.hash(password, 10);
      const user = new User({ username, email, password: hashedPassword, role });

      await user.save();
      const token = jwt.sign({ id: user._id, role: user.role }, process.env.JWT_SECRET, { expiresIn: "1h" });

      res.status(201).json({ message: "User registered successfully", id: user._id, token, role: user.role });
    } catch (error) {
      console.error("Registration error:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Login
app.post("/api/login", async (req, res) => {
  const { email, password } = req.body;
  try {
    const user = await User.findOne({ email });
    if (!user) {
      return res.status(400).json({ message: "User doesn't exist" });
    }
    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) {
      return res.status(400).json({ message: "Invalid credentials" });
    }

    const token = jwt.sign(
      { id: user._id, role: user.role },
      process.env.JWT_SECRET,
      { expiresIn: "1h" }
    );

    res.cookie("token", token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: "none",
      domain: process.env.NODE_ENV === 'production' ? '.amsoljobs.africa' : 'localhost',
    });

    res.json({
      message: "Login successful!",
      id: user._id,
      token,
      role: user.role,
    });
  } catch (error) {
    console.error("Login error:", error);
    res.status(500).json({ message: "Server error" });
  }
});

app.post("/logout", (req, res) => {
  res.clearCookie("authToken", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: 'Strict',
  });
  res.clearCookie("token", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: 'Strict',
  });
  res.clearCookie("userId", {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: 'Strict',
  });
  res.json({ message: "Logged out successfully" });
});

// Read user by ID endpoint
app.get("/api/users/:id", async (req, res) => {
  try {
    const user = await User.findById(req.params.id).select("-password");
    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }
    res.json(user);
  } catch (error) {
    console.error("Error fetching user by ID:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Delete user by ID
app.delete(
  "/api/users/:id",
  async (req, res) => {
    try {
      const user = await User.findById(req.params.id);
      if (!user) {
        return res.status(404).json({ message: "User not found" });
      }
      await user.remove();
      res.json({ message: "User deleted successfully" });
    } catch (error) {
      console.error("Error deleting user:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

//Get all users with pagination GET /api/users?page=1&limit=10
app.get("/api/users",auth, async (req, res) => {
  const { page = 1, limit = 10 } = req.query;
  const pageNumber = parseInt(page, 10);
  const pageSize = parseInt(limit, 10);
  try {
    const users = await User.find()
      .select("-password")
      .limit(pageSize)
      .skip((pageNumber - 1) * pageSize);
    const totalUsers = await User.countDocuments();
    res.json({
      total: totalUsers,
      page: pageNumber,
      limit: pageSize,
      users,
    });
  } catch (error) {
    console.error("Error fetching users:", error);
    res.status(500).json({ message: "Server error" });
  }
});

app.put("/api/profile/:userId", async (req, res) => {
  try {
    // Find the application data based on the user ID
    const applicationData = await Application.findOne({ userId: req.params.userId });

    if (!applicationData) {
      return res.status(404).json({ message: "Application data not found" });
    }
    const {
      first_name,
      last_name,
      phone,
      address,
      email,
      bio,
      work_experience, 
      education, 
      certifications, 
      skills,
    } = applicationData;

    // Update the user profile with all fields from the application data
    const updatedUser = await User.findByIdAndUpdate(
      req.params.userId,
      {
        "profile.first_name": first_name,
        "profile.last_name": last_name,
        "profile.phone": phone,
        "profile.address": address,
        "profile.email": email,
        "profile.bio": bio,
        "profile.work_experience": work_experience,
        "profile.education": education,
        "profile.certifications": certifications,
        "profile.skills": skills,
      },
      { new: true }
    );

    if (!updatedUser) {
      return res.status(404).json({ message: "User not found" });
    }

    res.json({ message: "Profile updated successfully", user: updatedUser });
  } catch (error) {
    console.error("Error updating user profile:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Update user profile endpoint (supports DOCX, PDF, and JSON)
app.put("/api/profile/resume/:id", upload.single("file"), async (req, res) => {
  try {
    const userId = req.params.id;
    const file = req.file;
    let extractedText;
    let profileDetails;

    if (file) {
      // Handle PDF file
      if (file.mimetype === "application/pdf") {
        extractedText = await pdfParse(file.buffer);
      }

      // Handle DOCX files
      if (file.mimetype === "application/vnd.openxmlformats-officedocument.wordprocessingml.document") {
        const result = await mammoth.extractRawText({ buffer: file.buffer });
        extractedText = result.value;
      }

      if (!extractedText) {
        return res.status(400).json({ message: "Unsupported file type" });
      }

      // Process extracted text (e.g., profile details extraction)
      profileDetails = extractProfileDetails(extractedText); // Ensure this function is defined

      // Store the file in GridFS
      const uploadStream = gfsBucket.openUploadStream(file.originalname, { contentType: file.mimetype });
      uploadStream.end(file.buffer); // Use buffer directly from memory storage

      // Update user profile
      const updatedUser = await User.findByIdAndUpdate(
        userId,
        { profile: profileDetails },
        { new: true }
      );

      if (!updatedUser) {
        return res.status(404).json({ message: "User not found" });
      }

      res.json({
        message: "Profile updated successfully",
        profile: updatedUser.profile,
      });
    } else {
      return res.status(400).json({ message: "No file uploaded" });
    }
  } catch (error) {
    console.error("Error updating profile:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Create Job Endpoint
app.post("/api/jobs", async (req, res) => {
  const {
    created_by,
    title,
    description,
    salary_range,
    job_status,
    category_id,
  } = req.body;
  if (  !title || !description || !salary_range ) {
    return res.status(400).json({ message: "All fields are required." });
  }
  if (salary_range.min === undefined || salary_range.max === undefined) {
    return res
      .status(400)
      .json({ message: "Salary range must include min and max." });
  }
  try {
    const newJob = new Job({
      created_by,
      title,
      description,
      salary_range,
      job_status,
      category_id,
    });
    const savedJob = await newJob.save();
    res.status(201).json(savedJob);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
});

// Get all jobs with pagination GET /api/jobs?page=1&limit=10
app.get("/api/jobs", async (req, res) => {
  const { page = 1, limit = 10 } = req.query;
  const pageNumber = parseInt(page, 10);
  const pageSize = parseInt(limit, 10);

  try {
    const jobs = await Job.find()
      .populate({
        path: "created_by",
        select: "-password",
      })
      .populate("category_id")
      .limit(pageSize)
      .skip((pageNumber - 1) * pageSize);

    const totalJobs = await Job.countDocuments();

    res.json({
      total: totalJobs,
      page: pageNumber,
      limit: pageSize,
      jobs,
    });
  } catch (error) {
    console.error("Error fetching jobs:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Get jobs by category
app.get("/api/jobs/category/:categoryId", async (req, res) => {
  const { categoryId } = req.params;
  try {
    const jobs = await Job.find({ category_id: categoryId }).populate(
      "created_by category_id"
    );
    if (!jobs.length) {
      return res
        .status(404)
        .json({ message: "No jobs found for this category." });
    }
    res.json(jobs);
  } catch (error) {
    console.error("Error fetching jobs by category:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Get a single job by ID
app.get("/api/jobs/:id", async (req, res) => {
  const { id } = req.params;
  try {
    const job = await Job.findById(id).populate("created_by category_id");
    if (!job) {
      return res.status(404).json({ message: "Job not found." });
    }
    res.json(job);
  } catch (error) {
    console.error("Error fetching job by ID:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Update job
app.put(
  "/api/jobs/:id",

  async (req, res) => {
    try {
      const updatedJob = await Job.findByIdAndUpdate(req.params.id, req.body, {
        new: true,
      });
      if (!updatedJob) {
        return res.status(404).json({ message: "Job not found" });
      }
      res.json(updatedJob);
    } catch (error) {
      console.error("Error updating job:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Delete job
app.delete(
  "/api/jobs/:id",

  async (req, res) => {
    try {
      const deletedJob = await Job.findByIdAndDelete(req.params.id);
      if (!deletedJob) {
        return res.status(404).json({ message: "Job not found" });
      }
      res.json({ message: "Job deleted successfully" });
    } catch (error) {
      console.error("Error deleting job:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Get all categories
app.get("/api/categories", async (req, res) => {
  try {
    const categories = await Category.find();
    res.json(categories);
  } catch (error) {
    console.error("Error fetching categories:", error);
    res.status(500).json({ message: "Server error" });
  }
});

// Get a single category by ID
app.get("/api/category/:id", async (req, res) => {
  const { id } = req.params;
  try {
    const category = await Category.findById(id);
    if (!category) {
      return res.status(404).json({ message: "Category not found." });
    }
    res.json(category);
  } catch (error) {
    console.error("Error fetching category by ID:", error);
    res.status(500).json({ message: "Server error", error: error.message });
  }
});

// Create Category Endpoint
app.post(
  "/api/categories",

  async (req, res) => {
    try {
      const { category_name } = req.body;
      if (!category_name) {
        return res.status(400).json({ message: "Category name is required." });
      }
      const newCategory = new Category({
        category_name,
      });
      await newCategory.save();
      res.status(201).json({
        message: "Category created successfully",
        category: newCategory,
      });
    } catch (error) {
      console.error("Error creating category:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Edit Category Endpoint
app.put(
  "/api/categories/:id",

  async (req, res) => {
    try {
      const { id } = req.params;
      const { category_name } = req.body;
      if (!category_name) {
        return res.status(400).json({ message: "Category name is required." });
      }
      const updatedCategory = await Category.findByIdAndUpdate(
        id,
        { category_name },
        { new: true, runValidators: true }
      );
      if (!updatedCategory) {
        return res.status(404).json({ message: "Category not found." });
      }
      res.status(200).json({
        message: "Category updated successfully",
        category: updatedCategory,
      });
    } catch (error) {
      console.error("Error updating category:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Delete Category Endpoint
app.delete(
  "/api/categories/:id",

  async (req, res) => {
    try {
      const { id } = req.params;
      const deletedCategory = await Category.findByIdAndDelete(id);
      if (!deletedCategory) {
        return res.status(404).json({ message: "Category not found." });
      }
      await Job.updateMany({ category: id }, { $set: { category: null } });
      res.status(200).json({ message: "Category deleted successfully." });
    } catch (error) {
      console.error("Error deleting category:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

app.post('/api/applications',auth, upload.single('cv'), async (req, res) => {
  try {
    console.log('Received form data:', req.body);
    console.log('Received file:', req.file);

    if (!req.file) {
      return res.status(400).json({ message: 'No CV file uploaded.' });
    }

    // Create a Readable stream from the file buffer
    const readableStream = new Readable();
    readableStream.push(req.file.buffer);
    readableStream.push(null);

    // Upload to GridFSBucket
    const uploadStream = gfsBucket.openUploadStream(req.file.originalname);
    readableStream.pipe(uploadStream);

    uploadStream.on('finish', async () => {
      const cvFileId = uploadStream.id;

      // Extract form fields
      const {
        firstName, lastName, secondName, idNumber, whatsAppNo,
        phoneNumber, email, age, nationality, location, specialization, positionapplied,hasDrivingLicence,
        PassportNo,
        academicLevel, salaryInfo, company1, position1, duration1,
        company2, position2, duration2, company3, position3, duration3
      } = req.body;

      // Create and save new application
      const newApplication = new Application({
        firstName,
        lastName,
        secondName,
        idNumber,
        whatsAppNo,
        phoneNumber,
        email,
        age,
        nationality,
        location,
        specialization,
        positionapplied,
        hasDrivingLicence,
        PassportNo,
        academicLevel,
        salaryInfo,
        cvFileId, 
        workExperience: [
          { company: company1, position: position1, duration: duration1 },
          { company: company2, position: position2, duration: duration2 },
          { company: company3, position: position3, duration: duration3 }
        ]
      });

      await newApplication.save();

      // Send confirmation email
      const subject = 'Application Received';
      const html = `
        <p>Dear ${firstName} ${lastName},</p>
        <p>Thank you for submitting your application. We wanted to let you know that we have successfully received your application, and it is currently being reviewed by our recruitment team.

If you are selected for the next stage of the process, we will reach out to you within the next two weeks. Should you not hear from us during this time, please consider your application unsuccessful on this occasion. We encourage you to keep an eye out for future opportunities that may be a fit for your skills and experience.

We appreciate your interest and wish you all the best in your job search.</p>
        <p>Best regards,<br>
        amsol.recruitment</p>
      `;

      try {
        await sendEmail(email, subject, html);
        res.status(200).json({ message: 'Application submitted successfully' });
      } catch (emailError) {
        console.error('Error sending confirmation email:', emailError);
        res.status(500).json({ message: 'Application submitted, but confirmation email failed.' });
      }
    });

    uploadStream.on('error', (err) => {
      console.error('Error uploading to GridFS:', err);
      res.status(500).json({ message: 'Error uploading CV file.' });
    });
  } catch (error) {
    console.error('Error handling application:', error);
    res.status(500).json({ message: 'Server error.' });
  }
});

// Route to retrieve the CV
app.get("/api/applications/cv/:id", async (req, res) => {
  const fileId = req.params.id;
  console.log("File ID received:", fileId);

  if (!gfsBucket) {
    return res.status(500).json({ err: "GridFSBucket is not initialized" });
  }

  try {
    // Find the file in GridFS
    const file = await mongoose.connection.db.collection("uploads.files").findOne({ _id: new mongoose.Types.ObjectId(fileId) });
    if (!file) {
      return res.status(404).json({ err: "No file exists" });
    }

    // Set headers and stream the file
    res.set({
      "Content-Type": file.contentType || "application/octet-stream",
      "Content-Disposition": `attachment; filename="${file.filename}"`,
    });

    const readStream = gfsBucket.openDownloadStream(file._id);
    readStream.pipe(res).on("error", (error) => {
      console.error("Error streaming file:", error);
      res.status(500).json({ err: "Error streaming file" });
    });
  } catch (error) {
    console.error("Error retrieving file:", error);
    res.status(500).json({ err: "Error retrieving file" });
  }
});

app.post("/api/applications/cv/:id", auth,async (req, res) => {
  const applicationId = req.params.id;

  if (!mongoose.isValidObjectId(applicationId)) {
    return res.status(400).json({ error: "Invalid application ID" });
  }

  const uploadStream = gfsBucket.openUploadStream(req.file.originalname, {
    contentType: req.file.mimetype,
  });
  req.file.stream.pipe(uploadStream);

  uploadStream.on("finish", async (uploadedFile) => {
    try {
      if (!uploadedFile._id) {
        return res.status(500).json({ error: "File upload failed" });
      }

      // Update the application with the new cvFileId
      await Application.findByIdAndUpdate(applicationId, { cvFileId: uploadedFile._id });
      res.status(200).json({ message: "CV uploaded successfully", cvId: uploadedFile._id });
    } catch (error) {
      console.error("Error updating application:", error);
      res.status(500).json({ error: "Error updating application" });
    }
  });

  uploadStream.on("error", (error) => {
    console.error("Upload stream error:", error);
    res.status(500).json({ error: "Error uploading file" });
  });
});

app.get("/api/user/:userId/cv", async (req, res) => {
  const userId = req.params.userId;

  try {
    // Find the user/application in the database
    const application = await Application.findOne({ userId });
    if (!application || !application.cvFileId) {
      return res.status(404).json({ error: "CV not found for this user" });
    }

    const fileId = application.cvFileId;

    if (!mongoose.isValidObjectId(fileId)) {
      return res.status(400).json({ error: "Invalid CV file ID format" });
    }

    const file = await mongoose.connection.db.collection("uploads.files").findOne({
      _id: new mongoose.Types.ObjectId(fileId),
    });

    if (!file) {
      return res.status(404).json({ error: "CV file not found" });
    }

    // Stream the file to the client
    res.set({
      "Content-Type": file.contentType || "application/octet-stream",
      "Content-Disposition": `attachment; filename="${file.filename}"`,
    });

    const readStream = gfsBucket.openDownloadStream(file._id);
    readStream.pipe(res).on("error", (error) => {
      console.error("Stream error:", error);
      res.status(500).json({ error: "Error streaming file" });
    });
  } catch (error) {
    console.error("Error retrieving CV:", error);
    res.status(500).json({ error: "Error retrieving CV" });
  }
});


// Route to get application details along with user profile
app.get("/applications/:applicationId", async (req, res) => {
  try {
    const application = await Application.findById(req.params.applicationId)
      .populate("applicant"); // Populate the applicant field

    if (!application) {
      return res.status(404).json({ message: "Application not found" });
    }

    res.json(application);
  } catch (error) {
    console.error("Error fetching application:", error);
    res.status(500).json({ message: "Server error" });
  }
});

const cache = new Map();

app.get("/api/applications", async (req, res) => {
  try {
    // Check if applications are already cached
    if (cache.has("applications")) {
      console.log("Serving from cache");
      return res.status(200).json(cache.get("applications"));
    }

    console.log("Fetching from database...");
    const applications = await Application.find().limit(10);

    // Store the data in cache
    cache.set("applications", applications);

    res.status(200).json(applications);
  } catch (error) {
    console.error("Error fetching applications:", error);
    res.status(500).json({ message: "Server Error", error });
  }
});


app.get("/api/applications/:Id", async (req, res) => {
  const { userId } = req.params;

  try {
    const application = await Application.findOne({ userId });

    if (!application) {
      return res.status(404).json({ message: "Application not found" });
    }

    res.status(200).json(application);
  } catch (error) {
    console.error("Error fetching application:", error);
    res.status(500).json({ message: "Server Error", error });
  }
});
app.patch('/api/applications/:id', async (req, res) => {
  const { id } = req.params;
  const { workExperience } = req.body;

  try {
    const application = await Application.findByIdAndUpdate(
      id,
      { workExperience },
      { new: true }
    );
    if (!application) {
      return res.status(404).json({ message: 'Application not found' });
    }
    res.json(application);
  } catch (error) {
    console.error('Error updating application:', error);
    res.status(500).json({ message: 'Internal Server Error' });
  }
});

// Update application status
app.put(
  "/api/applications/:id/status",

  async (req, res) => {
    const { status } = req.body;
    const validStatuses = ["applied", "interview", "hired", "rejected"];
    if (!validStatuses.includes(status)) {
      return res.status(400).json({ message: "Invalid status" });
    }
    try {
      const application = await Application.findById(req.params.id);
      if (!application) {
        return res.status(404).json({ message: "Application not found" });
      }
      application.status = status;
      await application.save();
      var html = "<p>Ola, your application status changed!</p>";
      await sendEmail(
        application.applicant,
        `Job application status changed to: ${status}`,
        html
      );
      res
        .status(200)
        .json({ message: "Application status updated successfully." });
    } catch (error) {
      console.error("Error updating application status:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// Delete application
app.delete(
  "/api/applications/:id",auth,

  async (req, res) => {
    try {
      const application = await Application.findByIdAndDelete(req.params.id);
      if (!application) {
        return res.status(404).json({ message: "Application not found" });
      }
      res.json({ message: "Application deleted successfully" });
    } catch (error) {
      console.error("Error deleting application:", error);
      res.status(500).json({ message: "Server error" });
    }
  }
);

// POST route for forgot password
app.post('/forgot-password', async (req, res) => {
  const { email } = req.body;

  // Check if the user exists
  const user = await User.findOne({ email });
  if (!user) {
    return res.status(404).json({ message: "User not found" });
  }

  // Create a reset token
  const token = jwt.sign({ id: user._id }, process.env.JWT_SECRET, { expiresIn: '1h' });

  // Create a reset link
  const resetLink = `https://amsol-api.onrender.com/reset-password/${token}`;

  // Set up Nodemailer
  const transporter = nodemailer.createTransport({
    service: 'Gmail',
    auth: {
      user: process.env.EMAIL_USER,
      pass: process.env.EMAIL_PASS,
    },
  });

  // Send the email
  const mailOptions = {
    from: process.env.EMAIL_USER,
    to: email,
    subject: 'Password Reset',
    text: `You requested a password reset. Click the link to reset your password: ${resetLink}`,
  };

  try {
    await transporter.sendMail(mailOptions);
    res.json({ message: "Reset link sent!" });
  } catch (error) {
    console.error("Error sending email:", error);
    res.status(500).json({ message: "Error sending email" });
  }
});

app.post('/reset-password', async (req, res) => {
  const { token, password } = req.body;

  try {
    // Verify token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const userId = decoded.id;

    // Find user and update password
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    user.password = password; // Ensure to hash the password before saving
    await user.save();

    res.json({ message: "Password reset successful!" });
  } catch (error) {
    console.error("Error resetting password:", error);
    res.status(500).json({ message: "Failed to reset password" });
  }
});


// Endpoint to send notifications to all applicants
app.post("/api/notifications", async (req, res) => {
  const { subject } = req.body; // Only extract subject since content is hardcoded

  try {
    // Retrieve all applicants from the database
    const applications = await Application.find({}, "email firstName");

    // Check if there are applicants to notify
    if (applications.length === 0) {
      return res.status(404).json({ message: "No applicants found." });
    }

    // Send emails to each applicant
    const emailPromises = applications.map(async (applicant) => {
      const { firstName, email } = applicant;

      // Hardcoded HTML content
      const personalizedHtml = `
        <div style="font-family: Arial, sans-serif; background-color: #f9f9f9; padding: 20px;">
          <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 20px; border-radius: 8px; box-shadow: 0 0 10px rgba(0, 0, 0, 0.1);">
            <h2 style="color: #333;">Hello ${firstName},</h2>
            <p style="font-size: 16px; line-height: 1.6; color: #555;">
              Click the link below to update your profile and be linked to available jobs:
            </p>
            <a href="https://your-website.com/update-profile" 
               style="display: inline-block; padding: 10px 15px; margin-top: 10px; background-color: #4CAF50; color: white; text-decoration: none; border-radius: 5px;">
              Update Profile
            </a>
            <p style="font-size: 14px; color: #888; margin-top: 20px;">Thank you,<br>The Team</p>
          </div>
        </div>`;

      // Send the email
      await sendEmail(email, subject, personalizedHtml);
    });

    // Wait for all emails to be sent
    await Promise.all(emailPromises);

    res.status(200).json({ message: "Emails sent to all applicants." });
  } catch (error) {
    console.error("Error sending notifications:", error);
    res.status(500).json({ message: "Error sending notifications." });
  }
});

// Start the server
// const PORT = process.env.PORT || 5000;
// app.listen(PORT, () => {
//   console.log(`Server running on port ${PORT}`);
// });
